<?php

namespace App\Http\Controllers;

use App\Http\Requests\ProductCategoryRequest;
use App\Models\ProductCategory;
use App\Models\ProductCategoryPercentage;
use App\Models\ProductCategoryPercentageDetail;
use Inertia\Inertia;

class ProductCategoryController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $query = ProductCategory::where('status', '1')
            ->latest('products_category.id');

        if (request()->input('name')) {
            $query->where('name', 'LIKE', '%' . request()->input('name') . '%');
        }

        if (request()->input('description')) {
            $query->where('description', 'LIKE', '%' . request()->input('description') . '%');
        }

        $categories = $query->paginate(10)->withQueryString();

        return Inertia::render('ProductsCategory/Index', [
            'categories' => $categories
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(ProductCategoryRequest $request)
    {
        \DB::transaction(function () use ($request) {
            $product_category = new ProductCategory($request->input());
            $product_category->status = 1;
            $product_category->save();

            foreach ($request->input('percentages') as $percentage_key => $value_key) {
                $percentage = new ProductCategoryPercentage(['product_category_id' => $product_category->id, 'client_category_id' => $percentage_key]);
                $percentage->status = 1;
                $percentage->save();

                foreach ($value_key as $percentage_detail_key => $value_percentage_detail) {
                    $percentage_detail = new ProductCategoryPercentageDetail(['percentage_id' => $percentage->id, 'currency_id' => $percentage_detail_key, 'percentage' => $value_percentage_detail]);
                    $percentage_detail->status = 1;
                    $percentage_detail->save();
                }
            }
        });

        return redirect('products_category');
    }

    /**
     * Display the specified resource.
     */
    public function show(ProductCategory $products_category)
    {
        return $products_category;
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(ProductCategoryRequest $request, ProductCategory $products_category)
    {
        \DB::transaction(function () use ($request, $products_category) {
            $products_category->update($request->input());

            $existingCurrencies = ProductCategoryPercentageDetail::whereIn('percentage_id', $products_category->percentages_relation->pluck('id'))->pluck('currency_id')->toArray();

            foreach ($request->input('percentages') as $percentage_key => $value_key) {
                $percentage = $products_category->percentages_relation()->updateOrCreate(
                    ['client_category_id' => $percentage_key],
                    ['status' => 1]
                );

                foreach ($value_key as $percentage_detail_key => $value_percentage_detail) {
                    if (!in_array($percentage_detail_key, $existingCurrencies)) {
                        $percentage_detail = new ProductCategoryPercentageDetail(['percentage_id' => $percentage->id, 'currency_id' => $percentage_detail_key, 'percentage' => $value_percentage_detail]);
                        $percentage_detail->status = 1;
                        $percentage_detail->save();
                    } else {
                        $existingPercentageDetail = ProductCategoryPercentageDetail::where('percentage_id', $percentage->id)->where('currency_id', $percentage_detail_key)->first();
                        if ($existingPercentageDetail) {
                            $existingPercentageDetail->update(['percentage' => $value_percentage_detail]);
                        } else {
                            $percentage_detail = new ProductCategoryPercentageDetail(['percentage_id' => $percentage->id, 'currency_id' => $percentage_detail_key, 'percentage' => $value_percentage_detail]);
                            $percentage_detail->status = 1;
                            $percentage_detail->save();
                        }
                    }
                }
            }
        });

        return redirect('products_category');
    }
    /**
     * Remove the specified resource from storage.
     */
    public function destroy(ProductCategory $products_category)
    {
        $products_category->status = 0;
        $products_category->save();
        $products_category->delete();
        return redirect('products_category');
    }
}
