<?php

namespace App\Http\Controllers;

use App\Models\Person;
use App\Models\Voucher;
use App\Models\VoucherDetail;
use Carbon\Carbon;
use DateTime;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Response;
use Inertia\Inertia;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;

class ReportController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        //
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }

    /**
     * Download the specified resource.
     */
    public function download(string $id)
    {
        //
    }

    public function reportCustomerLastSales(Request $request)
    {
        $dateFilter = $request->input('date');
        $type = $request->input('type');
        if (!$dateFilter) {
            $dateFilter = Carbon::today()->subDays(10)->toDateString();
        }

        $query = Voucher::select('person_id')
            ->selectRaw('MAX(issue_date) as issue_date')
            ->where('status', '1')
            ->with(['person' => function ($query) {
                $query->select('id', 'legal_name', 'document_number');
            }])
            ->havingRaw('MAX(issue_date) NOT BETWEEN ? AND ?', [$dateFilter, Carbon::today()])
            ->where('type_transaction', 'sales')
            ->groupBy('person_id')
            ->havingRaw('MAX(issue_date) >= ?', [Carbon::parse($dateFilter)->subDays(6)->toDateString()]);

        $chartData = $query->get();
        $customers = $query->paginate(2)->withQueryString();

        return Inertia::render('Report/CustomerLastSales/Index', [
            'customers' => $customers,
            'customersComplete' => $chartData,
            'dateFilter' => $dateFilter,
        ]);
    }

    public function downloadCustomerLastSales(Request $request)
    {
        $dateFilter = $request->input('date');
        /*     $data = Voucher::select('person_id')
                    ->selectRaw('MAX(issue_date) as issue_date')
                    ->where('status', '1')
                    ->with(['person' => function($query) {
                        $query->select('id', 'legal_name');
                    }])
                    ->whereBetween('issue_date', [$dateFilter, Carbon::today()])
                    ->where('type_transaction', 'sales')
                    ->groupBy('person_id')
                    ->get();*/

        $data = Voucher::select('person_id')
            ->selectRaw('MAX(issue_date) as issue_date')
            ->where('status', '1')
            ->with(['person' => function ($query) {
                $query->select('id', 'legal_name', 'document_number');
            }])
            ->where('type_transaction', 'sales')
            ->havingRaw('MAX(issue_date) NOT BETWEEN ? AND ?', [$dateFilter, Carbon::today()])
            ->groupBy('person_id')
            ->havingRaw('MAX(issue_date) >= ?', [Carbon::parse($dateFilter)->subDays(6)->toDateString()])
            ->get();

        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        $headerCellStyle = [
            'fill' => [
                'fillType' => Fill::FILL_SOLID,
                'color' => ['argb' => 'FFBF5700']
            ],
            'font' => [
                'bold' => true,
                'color' => ['argb' => 'FFFFFFFF'],
            ],
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_CENTER,
            ],
        ];

        $sheet->setCellValue('B3', 'Razon social');
        $sheet->setCellValue('C3', 'N° Documento');
        $sheet->setCellValue('D3', 'Ultima Compra');
        $sheet->getStyle('B3:D3')->applyFromArray($headerCellStyle);
        $sheet->getColumnDimension('B')->setWidth(60);
        $sheet->getColumnDimension('C')->setWidth(20);
        $sheet->getColumnDimension('D')->setWidth(20);

        $itemCellStyle = [
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_CENTER,
            ],
        ];

        $productCellStyle = [
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_LEFT,
            ],
        ];

        $row = 4;
        foreach ($data as $index => $detail) {
            $sheet->setCellValue('B' . $row, $detail->person->legal_name);
            $sheet->setCellValue('C' . $row, $detail->person->document_number);
            $sheet->setCellValue('D' . $row, Carbon::parse($detail->issue_date)->format('d/m/Y'));

            $row++;
        }

        $sheet->getStyle('B4:B' . ($row - 1))->applyFromArray($itemCellStyle);
        $sheet->getStyle('C4:C' . ($row - 1))->applyFromArray($productCellStyle);
        $sheet->getStyle('D4:D' . ($row - 1))->applyFromArray($productCellStyle);

        $writer = new Xlsx($spreadsheet);
        $filename = 'Reporte Clientes - Ultima Compra.xlsx';

        return Response::stream(
            function () use ($writer) {
                $writer->save('php://output');
            },
            200,
            [
                'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                'Content-Disposition' => "attachment; filename=\"{$filename}\"",
            ]
        );
    }

    public function chartCustomerLastSales(Request $request)
    {
        $dateFilter = $request->input('date');
        if (!$dateFilter) {
            $dateFilter = Carbon::today()->subDays(10)->toDateString();
        }

        $customers = Voucher::select('person_id')
            ->selectRaw('MAX(issue_date) as issue_date')
            ->where('status', '1')
            ->with(['person' => function ($query) {
                $query->select('id', 'legal_name', 'document_number');
            }])
            ->havingRaw('MAX(issue_date) NOT BETWEEN ? AND ?', [$dateFilter, Carbon::today()])
            ->where('type_transaction', 'sales')
            ->groupBy('person_id')->get();

        return response()->json([
            'customers' => $customers,
            'dateFilter' => $dateFilter
        ]);

    }

    public function reportCustomerSales(Request $request)
    {
        $personId = $request->input('id');
        $startFormatted = $request->input('startFormatted');
        $endFormatted = $request->input('endFormatted');

        if (!$startFormatted) {
            $startFormatted = Carbon::now('UTC')
                ->startOfMonth()
                ->format('Y-m-d');
        }

        if (!$endFormatted) {
            $endFormatted = Carbon::now('UTC')
                ->format('Y-m-d');
        }

        $query = Voucher::
        where('status', '1')
            ->with(['person' => function ($query) {
                $query->select('id', 'legal_name', 'document_number');
            }])
            ->where('type_transaction', 'sales')
            ->where('person_id', $personId)
            ->whereBetween('issue_date', [$startFormatted, $endFormatted]);

        $chartData = $query->get();
        $customers = $query->paginate(2)->withQueryString();

        $startFormatted = "$startFormatted";
        $date = DateTime::createFromFormat('Y-m-d', $startFormatted);
        $startDate = $date->format('Y/m/d');

        $endFormatted = "$endFormatted";
        $date = DateTime::createFromFormat('Y-m-d', $endFormatted);
        $endDate = $date->format('Y/m/d');

        return Inertia::render('Report/CustomerSales/Index', [
            'customers' => $customers,
            'customersComplete' => $chartData,
            'person_id' => $personId,
            'startFormatted' => $startDate,
            'endFormatted' => $endDate,
            'date' => [$startFormatted, $endFormatted],
        ]);
    }


    public function downloadCustomerSales(Request $request)
    {
        $personId = $request->input('id');
        $startFormatted = $request->input('startFormatted');
        $endFormatted = $request->input('endFormatted');

        $data = Voucher::
        where('status', '1')
            ->with(['person' => function ($query) {
                $query->select('id', 'legal_name', 'document_number');
            }])
            ->where('type_transaction', 'sales')
            ->where('person_id', $personId)
            ->whereBetween('issue_date', [$startFormatted, $endFormatted])
            ->get();

        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        $headerCellStyle = [
            'fill' => [
                'fillType' => Fill::FILL_SOLID,
                'color' => ['argb' => 'FFBF5700']
            ],
            'font' => [
                'bold' => true,
                'color' => ['argb' => 'FFFFFFFF'],
            ],
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_CENTER,
            ],
        ];

        $sheet->setCellValue('B3', 'Razon social');
        $sheet->setCellValue('C3', 'Numero Documento');
        $sheet->setCellValue('D3', 'Total');
        $sheet->setCellValue('E3', 'Fecha de Venta');
        $sheet->getStyle('B3:E3')->applyFromArray($headerCellStyle);
        $sheet->getColumnDimension('B')->setWidth(60);
        $sheet->getColumnDimension('C')->setWidth(20);
        $sheet->getColumnDimension('D')->setWidth(20);
        $sheet->getColumnDimension('E')->setWidth(20);

        $itemCellStyle = [
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_CENTER,
            ],
        ];

        $productCellStyle = [
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_LEFT,
            ],
        ];

        $row = 4;
        foreach ($data as $index => $detail) {
            $sheet->setCellValue('B' . $row, $detail->person->legal_name);
            $sheet->setCellValue('C' . $row, $detail->number);
            $sheet->setCellValue('D' . $row, $detail->total);
            $sheet->setCellValue('E' . $row, Carbon::parse($detail->issue_date)->format('d/m/Y'));

            $row++;
        }

        $sheet->getStyle('B4:B' . ($row - 1))->applyFromArray($itemCellStyle);
        $sheet->getStyle('C4:C' . ($row - 1))->applyFromArray($productCellStyle);
        $sheet->getStyle('D4:D' . ($row - 1))->applyFromArray($productCellStyle);
        $sheet->getStyle('E4:E' . ($row - 1))->applyFromArray($productCellStyle);

        $writer = new Xlsx($spreadsheet);
        $filename = 'Reporte Ventas por Cliente.xlsx';

        return Response::stream(
            function () use ($writer) {
                $writer->save('php://output');
            },
            200,
            [
                'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                'Content-Disposition' => "attachment; filename=\"{$filename}\"",
            ]
        );
    }


    public function reportProductSales(Request $request)
    {
        $categoryId = $request->input('id');
        $category = $request->input('category');
        $startFormatted = $request->input('startFormatted');
        $endFormatted = $request->input('endFormatted');

        if (!$startFormatted) {
            $startFormatted = Carbon::now('UTC')
                ->startOfMonth()
                ->format('Y-m-d');
        }

        if (!$endFormatted) {
            $endFormatted = Carbon::now('UTC')
                ->format('Y-m-d');
        }

        $query = Voucher::
        where('status', '1')
            ->with('details.product.category')
            ->with(['person' => function ($query) {
                $query->select('id', 'legal_name', 'document_number');
            }])
            ->where('type_transaction', 'sales')
            ->with(['details' => function ($query) {
                $query->select('id', 'quantity');
            }])
            ->with(['details.product' => function ($query) {
                $query->select('id', 'name');
            }])
            ->whereBetween('issue_date', [$startFormatted, $endFormatted]);
        if ($categoryId) {
            $query->whereHas('details.product.category', function ($q) use ($categoryId) {
                $q->where('id', $categoryId);
            });
        }


        $chartData = $query->get();
        $customers = $query->paginate(2)->withQueryString();

        $startFormatted = "$startFormatted";
        $date = DateTime::createFromFormat('Y-m-d', $startFormatted);
        $startDate = $date->format('Y/m/d');

        $endFormatted = "$endFormatted";
        $date = DateTime::createFromFormat('Y-m-d', $endFormatted);
        $endDate = $date->format('Y/m/d');

        return Inertia::render('Report/ProductSales/Index', [
            'customers' => $customers,
            'customersComplete' => $chartData,
            'category_id' => $categoryId,
            'category' => $category,
            'startFormatted' => $startDate,
            'endFormatted' => $endDate,
            'date' => [$startFormatted, $endFormatted],
        ]);
    }


    public function downloadProductSales(Request $request)
    {
        $categoryId = $request->input('id');
        $startFormatted = $request->input('startFormatted');
        $endFormatted = $request->input('endFormatted');

        if (!$startFormatted) {
            $startFormatted = Carbon::now('UTC')
                ->startOfMonth()
                ->format('Y-m-d');
        }

        if (!$endFormatted) {
            $endFormatted = Carbon::now('UTC')
                ->format('Y-m-d');
        }

        $query = Voucher::
        where('status', '1')
            ->with('details.product.category')
            ->with(['person' => function ($query) {
                $query->select('id', 'legal_name', 'document_number');
            }])
            ->where('type_transaction', 'sales')
            ->with(['details' => function ($query) {
                $query->select('id', 'quantity');
            }])
            ->with(['details.product' => function ($query) {
                $query->select('id', 'name');
            }])
            ->whereBetween('issue_date', [$startFormatted, $endFormatted]);
        if ($categoryId) {
            $query->whereHas('details.product.category', function ($q) use ($categoryId) {
                $q->where('id', $categoryId);
            });
        }


        $data = $query->get();

        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        $headerCellStyle = [
            'fill' => [
                'fillType' => Fill::FILL_SOLID,
                'color' => ['argb' => 'FFBF5700']
            ],
            'font' => [
                'bold' => true,
                'color' => ['argb' => 'FFFFFFFF'],
            ],
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_CENTER,
            ],
        ];

        $sheet->setCellValue('B3', 'Producto');
        $sheet->setCellValue('C3', 'Razon social');
        $sheet->setCellValue('D3', 'Numero Documento');
        $sheet->setCellValue('E3', 'Cantidad');
        $sheet->setCellValue('F3', 'Total');
        $sheet->setCellValue('G3', 'Fecha de Venta');
        $sheet->getStyle('B3:G3')->applyFromArray($headerCellStyle);
        $sheet->getColumnDimension('B')->setWidth(60);
        $sheet->getColumnDimension('C')->setWidth(20);
        $sheet->getColumnDimension('D')->setWidth(20);
        $sheet->getColumnDimension('E')->setWidth(20);

        $itemCellStyle = [
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_CENTER,
            ],
        ];

        $productCellStyle = [
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_LEFT,
            ],
        ];


        $row = 4;
        foreach ($data as $index => $detail) {
            $productNames = collect($detail->details)
                ->pluck('product.name')
                ->join(', ');
            $sheet->setCellValue('B' . $row, $productNames);
            $sheet->setCellValue('C' . $row, $detail->person->legal_name);
            $sheet->setCellValue('D' . $row, $detail->number);
            $totalQuantity = collect($detail->details)
                ->sum('quantity');
            $sheet->setCellValue('E' . $row, $totalQuantity);
            $sheet->setCellValue('F' . $row, $detail->total);
            $sheet->setCellValue('G' . $row, Carbon::parse($detail->issue_date)->format('d/m/Y'));
            $row++;
        }

        $sheet->getStyle('B4:B' . ($row - 1))->applyFromArray($itemCellStyle);
        $sheet->getStyle('C4:C' . ($row - 1))->applyFromArray($productCellStyle);
        $sheet->getStyle('D4:D' . ($row - 1))->applyFromArray($productCellStyle);
        $sheet->getStyle('E4:E' . ($row - 1))->applyFromArray($productCellStyle);

        $writer = new Xlsx($spreadsheet);
        $filename = 'Reporte Ventas por Familia.xlsx';

        return Response::stream(
            function () use ($writer) {
                $writer->save('php://output');
            },
            200,
            [
                'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                'Content-Disposition' => "attachment; filename=\"{$filename}\"",
            ]
        );
    }


    public function reportZoneSales(Request $request)
    {
        $ubigeo = $request->input('id');
        $department_id = $request->input('department_id');
        $province_id = $request->input('province_id');
        $paymentType = $request->input('paymentType');
        $peopleRol = $request->input('peopleRol');
        $people_role_id = $request->input('people_role_id');
        $startFormatted = $request->input('startFormatted');
        $endFormatted = $request->input('endFormatted');


        if (!$startFormatted) {
            $startFormatted = Carbon::now('UTC')
                ->subYear()
                ->startOfYear()
                ->format('Y-m-d');
        }

        if (!$endFormatted) {
            $endFormatted = Carbon::now('UTC')
                ->format('Y-m-d');
        }

        $query = Voucher::
        where('status', '1')
            ->with('details.product.category')
            ->with(['person' => function ($query) {
                $query->select('id', 'legal_name', 'document_number', 'ubigeo');
            }])
            ->where('type_transaction', 'sales')

            ->with(['details' => function ($query) {
                $query->select('id', 'quantity');
            }])
            ->with(['details.product' => function ($query) {
                $query->select('id', 'name');
            }])
            ->whereBetween('issue_date', [$startFormatted, $endFormatted]);
        if ($ubigeo) {
            $query->whereHas('person', function ($q) use ($ubigeo) {
                $q->where('ubigeo', $ubigeo);
            });
        }

        if ($paymentType != 'all') {

            $query->where('paymentType',$paymentType);
        }
        if ($people_role_id) {

            $query->where('people_role_id', $people_role_id);
        }



        $chartData = $query->get();
        $customers = $query->paginate(2)->withQueryString();

        $startFormatted = "$startFormatted";
        $date = DateTime::createFromFormat('Y-m-d', $startFormatted);
        $startDate = $date->format('Y/m/d');

        $endFormatted = "$endFormatted";
        $date = DateTime::createFromFormat('Y-m-d', $endFormatted);
        $endDate = $date->format('Y/m/d');


        return Inertia::render('Report/ZoneSales/Index', [
            'customers' => $customers,
            'customersComplete' => $chartData,
            'ubigeo' => $ubigeo,
            'department_id' => $department_id,
            'people_role_id' => $people_role_id,
            'peopleRol' => $peopleRol,
            'province_id' => $province_id,
            'paymentType' => $paymentType,
            'startFormatted' => $startDate,
            'endFormatted' => $endDate,
            'date' => [$startFormatted, $endFormatted],
            'dateFilter' => [$startDate . ' - ' . $endDate],
        ]);
    }

    public function downloadZoneSales(Request $request)
    {
        $ubigeo = $request->input('id');
        $department_id = $request->input('department_id');
        $province_id = $request->input('province_id');
        $paymentType = $request->input('paymentType');
        $peopleRol = $request->input('peopleRol');
        $people_role_id = $request->input('people_role_id');
        $startFormatted = $request->input('startFormatted');
        $endFormatted = $request->input('endFormatted');

        if (!$startFormatted) {
            $startFormatted = Carbon::now('UTC')
                ->subYear()
                ->startOfYear()
                ->format('Y-m-d');
        }

        if (!$endFormatted) {
            $endFormatted = Carbon::now('UTC')
                ->format('Y-m-d');
        }

        $query = Voucher::
        where('status', '1')
            ->with('details.product.category')
            ->with(['person' => function ($query) {
                $query->select('id', 'legal_name', 'document_number', 'ubigeo');
            }])
            ->where('type_transaction', 'sales')

            ->with(['details' => function ($query) {
                $query->select('id', 'quantity');
            }])
            ->with(['details.product' => function ($query) {
                $query->select('id', 'name');
            }])
            ->whereBetween('issue_date', [$startFormatted, $endFormatted]);
        if ($ubigeo) {
            $query->whereHas('person', function ($q) use ($ubigeo) {
                $q->where('ubigeo', $ubigeo);
            });
        }

        if ($paymentType != 'all') {

            $query->where('paymentType',$paymentType);
        }
        if ($people_role_id) {

            $query->where('people_role_id', $people_role_id);
        }

        $data = $query->get();

        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        $headerCellStyle = [
            'fill' => [
                'fillType' => Fill::FILL_SOLID,
                'color' => ['argb' => 'FFBF5700']
            ],
            'font' => [
                'bold' => true,
                'color' => ['argb' => 'FFFFFFFF'],
            ],
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_CENTER,
            ],
        ];

        $sheet->setCellValue('B3', 'Razon social');
        $sheet->setCellValue('C3', 'Numero Documento');
        $sheet->setCellValue('D3', 'Total');
        $sheet->setCellValue('E3', 'Fecha de Venta');
        $sheet->getStyle('B3:E3')->applyFromArray($headerCellStyle);
        $sheet->getColumnDimension('B')->setWidth(60);
        $sheet->getColumnDimension('C')->setWidth(20);
        $sheet->getColumnDimension('D')->setWidth(20);
        $sheet->getColumnDimension('E')->setWidth(20);

        $itemCellStyle = [
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_CENTER,
            ],
        ];

        $productCellStyle = [
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_LEFT,
            ],
        ];


        $row = 4;
        foreach ($data as $index => $detail) {
            $sheet->setCellValue('B' . $row, $detail->person->legal_name);
            $sheet->setCellValue('C' . $row, $detail->number);
            $sheet->setCellValue('D' . $row, $detail->total);
            $sheet->setCellValue('E' . $row, Carbon::parse($detail->issue_date)->format('d/m/Y'));
            $row++;
        }

        $sheet->getStyle('B4:B' . ($row - 1))->applyFromArray($itemCellStyle);
        $sheet->getStyle('C4:C' . ($row - 1))->applyFromArray($productCellStyle);
        $sheet->getStyle('D4:D' . ($row - 1))->applyFromArray($productCellStyle);
        $sheet->getStyle('E4:E' . ($row - 1))->applyFromArray($productCellStyle);

        $writer = new Xlsx($spreadsheet);
        $filename = 'Reporte Ventas por Zona.xlsx';

        return Response::stream(
            function () use ($writer) {
                $writer->save('php://output');
            },
            200,
            [
                'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                'Content-Disposition' => "attachment; filename=\"{$filename}\"",
            ]
        );
    }


    public function reportProviderPurchases(Request $request)
    {
        $personId = $request->input('id');
        $person = $request->input('person');
        $product_id = $request->input('product_id');
        $product = $request->input('product');
        $startFormatted = $request->input('startFormatted');
        $endFormatted = $request->input('endFormatted');

        if (!$startFormatted) {
            $startFormatted = Carbon::now('UTC')
                ->subYear()
                ->startOfYear()
                ->format('Y-m-d');
        }

        if (!$endFormatted) {
            $endFormatted = Carbon::now('UTC')
                ->format('Y-m-d');
        }

        $query = Voucher::
        where('status', '1')
            ->with(['person' => function ($query) {
                $query->select('id', 'legal_name', 'document_number');
            }])
            ->where('type_transaction', 'purchases')
            ->where('person_id', $personId)
            ->whereBetween('issue_date', [$startFormatted, $endFormatted]);


        if ($product_id) {
            $query->whereHas('details', function ($q) use ($product_id) {
                $q->where('product_id', $product_id);
            });
        }

        $chartData = $query->get();
        $providers = $query->paginate(2)->withQueryString();

        $startFormatted = "$startFormatted";
        $date = DateTime::createFromFormat('Y-m-d', $startFormatted);
        $startDate = $date->format('Y/m/d');

        $endFormatted = "$endFormatted";
        $date = DateTime::createFromFormat('Y-m-d', $endFormatted);
        $endDate = $date->format('Y/m/d');


        return Inertia::render('Report/ProviderPurchases/Index', [
            'providers' => $providers,
            'providersComplete' => $chartData,
            'person_id' => $personId,
            'person' => $person,
            'product_id' => $product_id,
            'product' => $product,
            'startFormatted' => $startDate,
            'endFormatted' => $endDate,
            'date' => [$startFormatted, $endFormatted],
        ]);
    }

    public function downloadProviderPurchases(Request $request)
    {
        $personId = $request->input('id');
        $person = $request->input('person');
        $startFormatted = $request->input('startFormatted');
        $endFormatted = $request->input('endFormatted');

        if (!$startFormatted) {
            $startFormatted = Carbon::now('UTC')
                ->subYear()
                ->startOfYear()
                ->format('Y-m-d');
        }

        if (!$endFormatted) {
            $endFormatted = Carbon::now('UTC')
                ->format('Y-m-d');
        }


        $query = Voucher::
        where('status', '1')
            ->with(['person' => function ($query) {
                $query->select('id', 'legal_name', 'document_number');
            }])
            ->where('type_transaction', 'purchases')
            ->where('person_id', $personId)
            ->whereBetween('issue_date', [$startFormatted, $endFormatted]);

        $data = $query->get();
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        $headerCellStyle = [
            'fill' => [
                'fillType' => Fill::FILL_SOLID,
                'color' => ['argb' => 'FFBF5700']
            ],
            'font' => [
                'bold' => true,
                'color' => ['argb' => 'FFFFFFFF'],
            ],
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_CENTER,
            ],
        ];

        $sheet->setCellValue('B3', 'Razon social');
        $sheet->setCellValue('C3', 'Numero Documento');
        $sheet->setCellValue('D3', 'Total');
        $sheet->setCellValue('E3', 'Fecha de Compra');
        $sheet->getStyle('B3:E3')->applyFromArray($headerCellStyle);
        $sheet->getColumnDimension('B')->setWidth(60);
        $sheet->getColumnDimension('C')->setWidth(20);
        $sheet->getColumnDimension('D')->setWidth(20);
        $sheet->getColumnDimension('E')->setWidth(20);

        $itemCellStyle = [
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_CENTER,
            ],
        ];

        $productCellStyle = [
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_LEFT,
            ],
        ];

        $row = 4;
        foreach ($data as $index => $detail) {
            $sheet->setCellValue('B' . $row, $detail->person->legal_name);
            $sheet->setCellValue('C' . $row, $detail->number);
            $sheet->setCellValue('D' . $row, $detail->total);
            $sheet->setCellValue('E' . $row, Carbon::parse($detail->issue_date)->format('d/m/Y'));

            $row++;
        }

        $sheet->getStyle('B4:B' . ($row - 1))->applyFromArray($itemCellStyle);
        $sheet->getStyle('C4:C' . ($row - 1))->applyFromArray($productCellStyle);
        $sheet->getStyle('D4:D' . ($row - 1))->applyFromArray($productCellStyle);
        $sheet->getStyle('E4:E' . ($row - 1))->applyFromArray($productCellStyle);

        $writer = new Xlsx($spreadsheet);
        $filename = 'Reporte Compras por Proveedor.xlsx';

        return Response::stream(
            function () use ($writer) {
                $writer->save('php://output');
            },
            200,
            [
                'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                'Content-Disposition' => "attachment; filename=\"{$filename}\"",
            ]
        );
    }


    public function reportSellerProfitSales(Request $request)
    {
        $ubigeo = $request->input('id');
        $department_id = $request->input('department_id');
        $province_id = $request->input('province_id');
        $paymentType = $request->input('paymentType');
        $peopleRol = $request->input('peopleRol');
        $people_role_id = $request->input('people_role_id');
        $startFormatted = $request->input('startFormatted');
        $endFormatted = $request->input('endFormatted');

        if (!$startFormatted) {
            $startFormatted = Carbon::now('UTC')
                ->subYear()
                ->startOfYear()
                ->format('Y-m-d');
        }

        if (!$endFormatted) {
            $endFormatted = Carbon::now('UTC')
                ->format('Y-m-d');
        }

        $query = Voucher::
        where('status', '1')
            ->with('details.product.category')
            ->with(['person' => function ($query) {
                $query->select('id', 'legal_name', 'document_number', 'ubigeo');
            }])
            ->where('type_transaction', 'sales')

            ->with(['details' => function ($query) {
                $query->select('id', 'quantity');
            }])
            ->with(['details.product' => function ($query) {
                $query->select('id', 'name');
            }])
            ->with('profit_details')
            ->whereBetween('issue_date', [$startFormatted, $endFormatted]);

        if ($paymentType != 'all') {
            $query->where('paymentType',$paymentType);
        }
        if ($people_role_id) {
            $query->where('people_role_id', $people_role_id);
        }

        $chartData = $query->get();
        $customers = $query->paginate(4)->withQueryString();

        $startFormatted = "$startFormatted";
        $date = DateTime::createFromFormat('Y-m-d', $startFormatted);
        $startDate = $date->format('Y/m/d');

        $endFormatted = "$endFormatted";
        $date = DateTime::createFromFormat('Y-m-d', $endFormatted);
        $endDate = $date->format('Y/m/d');


        return Inertia::render('Report/SellerProfitSales/Index', [
            'customers' => $customers,
            'customersComplete' => $chartData,
            'ubigeo' => $ubigeo,
            'department_id' => $department_id,
            'people_role_id' => $people_role_id,
            'peopleRol' => $peopleRol,
            'province_id' => $province_id,
            'paymentType' => $paymentType,
            'startFormatted' => $startDate,
            'endFormatted' => $endDate,
            'date' => [$startFormatted, $endFormatted],
            'dateFilter' => [$startDate . ' - ' . $endDate],
        ]);
    }

    public function downloadSellerProfitSales(Request $request)
    {
        $ubigeo = $request->input('id');
        $department_id = $request->input('department_id');
        $province_id = $request->input('province_id');
        $paymentType = $request->input('paymentType');
        $peopleRol = $request->input('peopleRol');
        $people_role_id = $request->input('people_role_id');
        $startFormatted = $request->input('startFormatted');
        $endFormatted = $request->input('endFormatted');


        if (!$startFormatted) {
            $startFormatted = Carbon::now('UTC')
                ->subYear()
                ->startOfYear()
                ->format('Y-m-d');
        }

        if (!$endFormatted) {
            $endFormatted = Carbon::now('UTC')
                ->format('Y-m-d');
        }

        $query = Voucher::
        where('status', '1')
            ->with('details.product.category')
            ->with(['person' => function ($query) {
                $query->select('id', 'legal_name', 'document_number', 'ubigeo');
            }])
            ->where('type_transaction', 'sales')

            ->with(['details' => function ($query) {
                $query->select('id', 'quantity');
            }])
            ->with(['details.product' => function ($query) {
                $query->select('id', 'name');
            }])
            ->with('profit_details')
            ->whereBetween('issue_date', [$startFormatted, $endFormatted]);
        if ($ubigeo) {
            $query->whereHas('person', function ($q) use ($ubigeo) {
                $q->where('ubigeo', $ubigeo);
            });
        }

        if ($paymentType != 'all') {

            $query->where('paymentType',$paymentType);
        }
        if ($people_role_id) {

            $query->where('people_role_id', $people_role_id);
        }

        $data = $query->get();

        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        $headerCellStyle = [
            'fill' => [
                'fillType' => Fill::FILL_SOLID,
                'color' => ['argb' => 'FFBF5700']
            ],
            'font' => [
                'bold' => true,
                'color' => ['argb' => 'FFFFFFFF'],
            ],
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_CENTER,
            ],
        ];

        $sheet->setCellValue('B3', 'Razon social');
        $sheet->setCellValue('C3', 'Numero Documento');
        $sheet->setCellValue('D3', 'Total');
        $sheet->setCellValue('E3', 'Ganancia');
        $sheet->setCellValue('F3', '% de Ganancia');
        $sheet->setCellValue('G3', 'Fecha de Venta');
        $sheet->getStyle('B3:G3')->applyFromArray($headerCellStyle);
        $sheet->getColumnDimension('B')->setWidth(60);
        $sheet->getColumnDimension('C')->setWidth(20);
        $sheet->getColumnDimension('D')->setWidth(20);
        $sheet->getColumnDimension('E')->setWidth(20);
        $sheet->getColumnDimension('F')->setWidth(20);
        $sheet->getColumnDimension('G')->setWidth(20);

        $itemCellStyle = [
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_CENTER,
            ],
        ];

        $productCellStyle = [
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_LEFT,
            ],
        ];

        $row = 4;
        $profitTotal = 0;
        $profitTotalPercentage = 0;
        foreach ($data as $index => $detail) {
            if(count($detail->profit_details) > 0){
                foreach ($detail->profit_details as $profit) {
                    $profitTotalPercentage += (float)$profit->percentage_total ?? 0;
                    $profitTotal += (float)$profit->profit_total ?? 0;
                }
            }

            $currencySymbol= $detail->currency_id == 1 ? 'S/ ' : '$ ';
            $sheet->setCellValue('B' . $row, $detail->person->legal_name);
            $sheet->setCellValue('C' . $row, $detail->number);
            $sheet->setCellValue('D' . $row,$currencySymbol . $detail->total);
            $sheet->setCellValue('E' . $row, $currencySymbol . $profitTotal);
            $sheet->setCellValue('F' . $row, $profitTotalPercentage . '%');
            $sheet->setCellValue('G' . $row, Carbon::parse($detail->issue_date)->format('d/m/Y'));
            $row++;
        }

        $sheet->getStyle('B4:B' . ($row - 1))->applyFromArray($itemCellStyle);
        $sheet->getStyle('C4:C' . ($row - 1))->applyFromArray($productCellStyle);
        $sheet->getStyle('D4:D' . ($row - 1))->applyFromArray($productCellStyle);
        $sheet->getStyle('E4:F' . ($row - 1))->applyFromArray($productCellStyle);

        $writer = new Xlsx($spreadsheet);
        $filename = 'Reporte Ventas por Zona.xlsx';

        return Response::stream(
            function () use ($writer) {
                $writer->save('php://output');
            },
            200,
            [
                'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                'Content-Disposition' => "attachment; filename=\"{$filename}\"",
            ]
        );
    }

}
