<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;

class Voucher extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'id',
        'number',
        'paymentType',
        'currency_id',
        'budget_id',
        'issue_date',
        'person_id',
        'people_role_id',
        'warehouse_id',
        'voucher_type_id',
        'discounts',
        'typeDiscount',
        'type_transaction',
        'subtotal',
        'igv',
        'total',
        'observations',
        'creation_user',
        'modification_user',
        'elimination_user',
        'status'
    ];

    protected $dates = ['deleted_at'];

    protected $appends = ['series', 'correlative'];

    public function setSubtotalAttribute($value)
    {
        $this->attributes['subtotal'] = round($value, 2);
    }

    public function setIgvAttribute($value)
    {
        $this->attributes['igv'] = round($value, 2);
    }


    public function setTotalAttribute($value)
    {
        $this->attributes['total'] = round($value, 2);
    }

    public function currency(): BelongsTo
    {
        return $this->belongsTo(Currency::class);
    }

    public function warehouse(): BelongsTo
    {
        return $this->belongsTo(warehouse::class);
    }

    public function person(): BelongsTo
    {
        return $this->belongsTo(Person::class);
    }

    public function budget(): BelongsTo
    {
        return $this->belongsTo(Budget::class);
    }

    public function voucher_type(): BelongsTo
    {
        return $this->belongsTo(VoucherType::class);
    }

    public function details(): HasMany
    {
        return $this->hasMany(VoucherDetail::class);
    }

    public function quotas(): HasMany
    {
        return $this->hasMany(Quota::class, 'document_id')->where('quotas.type_document', 'voucher');
    }

    protected function getSeriesAttribute()
    {
        if ($this->number) {
            $parts = explode("-", $this->number);

            return $parts[0];
        }

        return $this->getSeries($this->voucher_type_id);
    }

    protected function getCorrelativeAttribute()
    {
        if ($this->number) {
            $parts = explode("-", $this->number);

            return $parts[1];
        }

        return Voucher::where('voucher_type_id', $this->voucher_type_id)->count() + 1;
    }

    public function getSeries($voucher_type_id): string
    {
        $serie = Serie::where('voucher_type_id', $voucher_type_id)->first();

        if ($serie && $serie->value) {
            return $serie->value;
        }

        if($voucher_type_id == 1) {
            return 'F001';
        } else if($voucher_type_id == 2) {
            return 'B001';
        }

        return 'V001';
    }

    public function getCorrelative($voucher_type_id, $type_transaction): string
    {
        $currentSeries = $this->getSeries($voucher_type_id);
        $lastDocument = Voucher::where('number', 'like', $currentSeries . '-%')
            ->where('voucher_type_id', $voucher_type_id)
            ->where('type_transaction', $type_transaction)
            ->orderBy('id', 'desc')
            ->first();
        if ($lastDocument && $lastDocument->number) {
            $lastDocumentNumber = explode('-', $lastDocument->number)[1];
            $nextCorrelativeNumber = (int) $lastDocumentNumber + 1;
            $newCorrelative = str_pad($nextCorrelativeNumber, strlen($lastDocumentNumber), '0', STR_PAD_LEFT);

            return $newCorrelative;
        }

        return '0001';
    }

    public function getPeopleRoleByName($people_role_id)
    {
        return Person::join('people_role', 'people.id', '=', 'people_role.people_id')
            ->with('doctype')
            ->where('people_role.role_id', 2)
            ->where('people.status', 1)
            ->where('people_role.id', $people_role_id)
            ->select('people.*', 'people_role.id as people_role_id')
            ->latest('people.id')
            ->first();
    }

    public function people_role(): BelongsTo
    {
        return $this->belongsTo(peopleRole::class);
    }

    public function statementOfAccounts(): HasMany
    {
        return $this->hasMany(StatementOfAccount::class, 'voucher_id', 'id');
    }

    public function profit_details(): HasMany
    {
        return $this->hasMany(profitDetails::class);
    }


}
